<?php

/*
Plugin Name: Advanced Custom Fields: Currency Field
Plugin URI: http://wpcoders.io
Description: ACF field to add numbers by currency.
Version: 1.3.0
Author: Adam Fischer
Author URI: http://wpcoders.io
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
*/

// exit if accessed directly
if( ! defined( 'ABSPATH' ) ) exit;


// check if class already exists
if( !class_exists('acf_plugin_currency') ) :


class acf_plugin_currency {

    /*
    * The single instance of the class.
    */
    protected static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

	/*
	*  __construct
	*
	*  This function will setup the class functionality
	*
	*  @type	function
	*  @date	17/02/2016
	*  @since	1.0.0
	*
	*  @param	n/a
	*  @return	n/a
	*/

	function __construct() {

        // vars
        $this->settings_title = 'acf_currency_settings';

        $this->settings = array(
            'version'         => '1.3.0',
            'url'             => plugin_dir_url( __FILE__ ),
            'path'            => plugin_dir_path( __FILE__ ),
            'currencies'      => $this->get_currencies(),
            'number_format'   => $this->get_number_format_settings(),
            'auto_conversion' => $this->has_auto_conversion(),
            'rates'           => $this->get_exchange_rates()
        );

		// set text domain
		// https://codex.wordpress.org/Function_Reference/load_plugin_textdomain
		load_plugin_textdomain( 'acf-currency', false, plugin_basename( dirname( __FILE__ ) ) . '/lang' );


		// include field
		add_action( 'acf/include_field_types', 	array( $this, 'include_field_types' ) ); // v5

        // options page
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) , 100);
        add_action( 'admin_init', array( $this, 'settings_init' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_settings_assets' ), 100);

        // update exchange rates
        add_action( 'wp_ajax_acf_currency_update_exchange_rates', array( $this, 'update_exchange_rates_ajax' ) );
        add_action( 'wp_ajax_nopriv_acf_currency_update_exchange_rates', array( $this, 'update_exchange_rates_ajax' ) );
        add_action( 'init', array( $this, 'update_exchange_rates' ) );

        // utility functions
        if ( ! function_exists( 'acf_currency_get_currencies' ) ) {
            function acf_currency_get_currencies() {
                return acf_plugin_currency::instance()->get_currencies();
            }
        }

        if ( ! function_exists( 'acf_currency_get_number_format_settings' ) ) {
            function acf_currency_get_number_format_settings() {
                return acf_plugin_currency::instance()->get_number_format_settings();
            }
        }

        if ( ! function_exists( 'acf_currency_number_format' ) ) {
            function acf_currency_number_format( $number ) {
                $settings = acf_currency_get_number_format_settings();
                return number_format( $number, $settings['decimals'], $settings['dec_point'], $settings['thousands_sep'] );
            }
        }

        if ( ! function_exists( 'acf_currency_has_auto_conversion' ) ) {
            function acf_currency_has_auto_conversion() {
                return acf_plugin_currency::instance()->has_auto_conversion();
            }
        }

        if ( ! function_exists( 'acf_currency_get_exchange_rates' ) ) {
            function acf_currency_get_exchange_rates() {
                return acf_plugin_currency::instance()->get_exchange_rates();
            }
        }

        if ( ! function_exists( 'acf_currency_get_api_exchange_rates' ) ) {
            function acf_currency_get_api_exchange_rates( $date = 'latest', $default_code = '', $specific_exchange_rates_codes = [] ) {
                return acf_plugin_currency::instance()->get_api_exchange_rates( $date, $default_code, $specific_exchange_rates_codes );
            }
        }
	}


	/*
	*  include_field_types
	*
	*  This function will include the field type class
	*
	*  @type	function
	*  @date	17/02/2016
	*  @since	1.0.0
	*
	*  @param	$version (int) major ACF version. Defaults to false
	*  @return	n/a
	*/

	function include_field_types( $version = false ) {

		// support empty $version
		if( !$version ) $version = 5;

		// include
		include_once('fields/acf-currency-v' . $version . '.php');

	}


    /*
    * This functions will create options page
    */

    function enqueue_settings_assets() {

        // vars
        $url = $this->settings['url'];
        $version = $this->settings['version'];

        // register & include JS
        wp_register_script( 'acf-input-currency-settings', "{$url}assets/js/settings.js", array( 'jquery', 'jquery-ui-sortable' ), $version );
        wp_enqueue_script( 'acf-input-currency-settings' );

        // register & include CSS
        wp_register_style( 'acf-input-currency-settings', "{$url}assets/css/settings.css", false, $version );
        wp_enqueue_style( 'acf-input-currency-settings' );
    }

    function add_admin_menu() {

        add_submenu_page(
            'edit.php?post_type=acf-field-group',
            __( 'Currency Field Settings', 'acf-currency' ),
            __( 'Currency', 'acf-currency' ),
            'manage_options',
            'acf-currency-settings',
            array( $this, 'options_page' )
        );

    }

    function settings_init() {

        register_setting(
            $this->settings_title,
            $this->settings_title,
            [ $this, 'sanitize_settings' ]
        );

        add_settings_section(
            $this->settings_title . '_currencies',
            '',
            '__return_false',
            $this->settings_title
        );

        add_settings_field(
            $this->settings_title . '_currencies',
            __( 'Currencies', 'acf-currency' ),
            array( $this, 'currencies_field_render' ),
            $this->settings_title,
            $this->settings_title . '_currencies',
            array(
                "field_name"  => $this->settings_title . '[currencies]',
                "setting_key" => 'currencies',
            )
        );

        add_settings_field(
            $this->settings_title . '_auto_conversion',
            __( 'Auto Conversion', 'acf-currency' ),
            array( $this, 'select_field_render' ),
            $this->settings_title,
            $this->settings_title . '_currencies',
            array(
                "field_name"  => $this->settings_title . '[auto_conversion]',
                "field_id"    => $this->settings_title . '_auto_conversion',
                "setting_key" => 'auto_conversion',
                "field_class" => '',
                "choices"     => array(
                    'enabled'  => __( 'Enabled', 'acf-currency' ),
                    'disabled' => __( 'Disabled', 'acf-currency' ),
                ),
                "default"     => 'disabled',
                "description" => ''
            )
        );

        add_settings_field(
            $this->settings_title . '_fixer_api_key',
            __( 'Fixer API Access Key', 'acf-currency' ),
            array( $this, 'text_field_render' ),
            $this->settings_title,
            $this->settings_title . '_currencies',
            array(
                "field_name"  => $this->settings_title . '[fixer_api_key]',
                "field_id"    => $this->settings_title . '_fixer_api_key',
                "setting_key" => 'fixer_api_key',
                "field_class" => 'regular-text',
                "default"     => '',
                "description" => '<a href="https://fixer.io/" target="_blank">https://fixer.io/</a>'
            )
        );

        add_settings_section(
            $this->settings_title . '_number_format',
            __( 'Number Format', 'acf-currency' ),
            '__return_false',
            $this->settings_title
        );

        add_settings_field(
            $this->settings_title . '_decimals',
            __( 'Decimals', 'acf-currency' ),
            array( $this, 'number_field_render' ),
            $this->settings_title,
            $this->settings_title . '_number_format',
            array(
                "field_name"  => $this->settings_title . '[decimals]',
                "field_id"    => $this->settings_title . '_decimals',
                "setting_key" => 'decimals',
                "field_class" => 'small-text',
                "default"     => 0,
                "description" => __( 'Sets the number of decimal points.', 'acf-currency' )
            )
        );

        add_settings_field(
            $this->settings_title . '_dec_point',
            __( 'Decimal Point', 'acf-currency' ),
            array( $this, 'text_field_render' ),
            $this->settings_title,
            $this->settings_title . '_number_format',
            array(
                "field_name"  => $this->settings_title . '[dec_point]',
                "field_id"    => $this->settings_title . '_dec_point',
                "setting_key" => 'dec_point',
                "field_class" => 'small-text',
                "default"     => '.',
                "description" => __( 'Sets the separator for the decimal point.', 'acf-currency' )
            )
        );

        add_settings_field(
            $this->settings_title . '_thousands_sep',
            __( 'Thousands Separator', 'acf-currency' ),
            array( $this, 'text_field_render' ),
            $this->settings_title,
            $this->settings_title . '_number_format',
            array(
                "field_name"  => $this->settings_title . '[thousands_sep]',
                "field_id"    => $this->settings_title . '_thousands_sep',
                "setting_key" => 'thousands_sep',
                "field_class" => 'small-text',
                "default"     => ',',
                "description" => __( 'Sets the thousands separator.', 'acf-currency' )
            )
        );

    }

    function sanitize_settings( $input_fields ) {

        $valid_fields = get_option( $this->settings_title, array() );
        $suffix = $this->get_language_suffix();

        foreach ( $input_fields as $key => $val ) {

            $valid_key = $key . $suffix;

            // Currencies
            if ( $key === 'currencies' ) {
                $valid_fields[ $valid_key ] = array();

                if ( ! empty ( $currencies = $input_fields[ $key ] ) ) {
                    foreach ( (array) $currencies as $currency ) {
                        if ( ! empty( $currency['code'] ) && ! empty( $currency['sign'] ) ) {
                            $valid_fields[ $valid_key ][] = $currency;
                        }
                    }
                }
            }

            // Text & Select
            switch ( $key ) {
                case 'dec_point':
                case 'thousands_sep':
                case 'auto_conversion':
                case 'fixer_api_key':
                    $valid_fields[ $valid_key ] = strip_tags( stripslashes( trim( $val, '\t\n\r\0\x0B' ) ) );
                    break;
            }

            // Number
            switch ( $key ) {
                case 'decimals':
                    $valid_fields[ $valid_key ] = absint( $val );
                    break;
            }

            // Exchange rates
            if ( isset( $input_fields['auto_conversion'] ) && $input_fields['auto_conversion'] === 'enabled' ) {

                if ( ! empty( $valid_fields['currencies'] ) ) {
                    $default_code                  = '';
                    $specific_exchange_rates_codes = [];

                    foreach ( $valid_fields['currencies'] as $key => $currency ) {

                        if ( $key === 0 ) {
                            $default_code = strtoupper( $currency['code'] );
                        } else {
                            $specific_exchange_rates_codes[] = strtoupper( $currency['code'] );
                        }
                    }

                    if ( ! empty( $default_code ) && ! empty( $specific_exchange_rates_codes ) ) {
                        $rates = $this->get_api_exchange_rates( 'latest', $default_code, $specific_exchange_rates_codes );

                        if ( ! empty( $rates ) ) {
                            $valid_fields['rates'] = $rates;
                        }

                        $valid_fields['rates_last_update'] = date( 'Y-m-d H:i:s' );
                    }
                }
            }
        }

        return $valid_fields;
    }

    function currencies_field_render( $args ) {

        $field_value = $this->get_setting( $args['setting_key'] );
        ?>

        <div class="currency-wrap <?= empty( $field_value ) ? 'hidden' : ''; ?>">
            <p class="currency-container">
                <label><?= __( 'Code', 'acf-currency' ); ?></label>
                <input type="text" class="small-text" name="<?= $args['field_name'] ?>[0][code]"
                       value="<?= empty( $field_value[0]['code'] ) ? '' : $field_value[0]['code']; ?>">

                <label><?= __( 'Sign', 'acf-currency' ); ?></label>
                <input type="text" class="small-text" name="<?= $args['field_name'] ?>[0][sign]"
                       value="<?= empty( $field_value[0]['sign'] ) ? '' : $field_value[0]['sign']; ?>">

                <a href="#" class="remove-currency"><?= __( 'Remove', 'acf-currency' ); ?></a>
            </p>

            <?php if ( is_array( $field_value ) && ! empty( $field_value[1] ) ):
                foreach ( array_slice( $field_value, 1 ) as $key => $field_value ): $key ++; ?>
                    <p class="currency-container">
                        <label><?= __( 'Code', 'acf-currency' ); ?></label>
                        <input type="text" class="small-text" name="<?= $args['field_name'] ?>[<?= $key; ?>][code]"
                               value="<?= empty( $field_value['code'] ) ? '' : $field_value['code']; ?>">

                        <label><?= __( 'Sign', 'acf-currency' ); ?></label>
                        <input type="text" class="small-text" name="<?= $args['field_name'] ?>[<?= $key; ?>][sign]"
                               value="<?= empty( $field_value['sign'] ) ? '' : $field_value['sign']; ?>">

                        <a href="#" class="remove-currency"><?= __( 'Remove', 'acf-currency' ); ?></a>
                    </p>
                <?php endforeach;
            endif; ?>
        </div>

        <div class="add-currency-button-wrap">
            <button type="button" name="add_currency" id="add_currency" class="button button-secondary">
                <?= __( 'Add Currency', 'acf-currency' ); ?>
            </button>
        </div>
        <?php
    }

    function text_field_render( $args ) {

        $field_value = $this->get_setting( $args['setting_key'] );

        if ( empty( $field_value ) ) {
            $field_value = $args['default'];
        }

        ?>
        <input type="text" class="<?= $args['field_class']; ?>" id="<?= $args['field_id']; ?>"
               name="<?= $args['field_name'] ?>" value="<?= esc_attr( $field_value ); ?>">
        <?php

        if ( isset( $args['description'] ) ):
            ?>
            <p class="description"><?= $args['description']; ?></p>
            <?php
        endif;

    }

    function select_field_render( $args ) {

        $field_value = $this->get_setting( $args['setting_key'] );

        if ( empty( $field_value ) ) {
            $field_value = $args['default'];
        }

        ?>
        <select name="<?=  $args['field_name'] ?>" id="<?=  $args['field_id']; ?>" class="<?=  $args['field_class']; ?>">
            <?php foreach ( $args['choices'] as $key => $value ): ?>
                <option class="level-0" value="<?=  $key; ?>" <?php selected( $key, $field_value ); ?>>
                    <?=  $value; ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php

        if ( isset( $args['description'] ) ):
            ?>
            <p class="description"><?= $args['description']; ?></p>
        <?php
        endif;

    }

    function number_field_render( $args ) {

        $field_value = $this->get_setting( $args['setting_key'] );

        if ( empty( $field_value ) ) {
            $field_value = $args['default'];
        }

        ?>
        <input type="number" value="<?= esc_attr( $field_value ); ?>"
               id="<?= $args['field_id']; ?>" class="<?= $args['field_class']; ?>" name="<?= $args['field_name']; ?>">
        <?php

        if ( isset( $args['description'] ) ):
            ?>
            <p class="description"><?= $args['description']; ?></p>
            <?php
        endif;

    }

    function exchange_rates_table() {

        $rates = $this->get_exchange_rates();

        if ( ! empty( $rates ) && $this->has_auto_conversion() ):
            ?>
            <h2><?= __( 'Exchange Rates', 'acf-currency' ); ?></h2>
            <div class="form-table">
                <table class="exchange-rates-table" border="1">
                    <?php foreach ( $rates as $code => $rate ): ?>
                        <tr>
                            <td><strong><?= $code; ?></strong></td>
                            <td><?= $rate; ?></td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            </div>
        <?php
        endif;
    }

    function options_page() {

        ?>

        <div class="wrap">

            <h1><?= __( 'Currency Field Settings', 'acf-currency' ); ?></h1>

            <form action='options.php' method='post'>

                <?php
                settings_fields( $this->settings_title );
                do_settings_sections( $this->settings_title );
                submit_button();
                ?>

            </form>

            <?php $this->exchange_rates_table(); ?>

        </div>

        <?php

    }


    /*
    * Get plugin setting value
    */

    function get_setting( $setting_key ) {

        $settings = get_option( $this->settings_title );
        $suffix = $this->get_language_suffix();

        if ( isset( $settings[ $setting_key . $suffix ] ) ) {

            return $settings[ $setting_key . $suffix ];
        }

        if ( isset( $settings[ $setting_key ] ) ) {

            return $settings[ $setting_key ];
        }

        return false;
    }

    /*
     * Get language
     */

    function get_language_suffix() {
        global $sitepress;

        if ( empty( $sitepress ) ) {
            return '';
        }

        $default = $sitepress->get_default_language();
        $current = $sitepress->get_current_language();

        if ( $default !== $current ) {
            return '_' . $sitepress->get_current_language();
        }

        return '';
    }

    /*
    * Get currencies
    */

    function get_currencies() {

        return $this->get_setting( 'currencies' );
    }

    /*
    * Get number format settings
    */

    function get_number_format_settings() {

        return array(
            'decimals'      => $this->get_setting( 'decimals' ),
            'dec_point'     => $this->get_setting( 'dec_point' ),
            'thousands_sep' => $this->get_setting( 'thousands_sep')
        );
    }

    /*
    * Get auto currency conversion status
    */

    function has_auto_conversion() {

        if ( $this->get_setting( 'auto_conversion' ) === 'enabled' ) {
            return true;
        }

        return false;
    }

    /*
    * Get exchange rates
    */

    function get_exchange_rates() {

        return $this->get_setting( 'rates' );
    }

    /*
    * Get api exchange rates
    */

    function get_api_exchange_rates( $date = 'latest', $default_code = '', $specific_exchange_rates_codes = [] ) {

        $rates          = [];
        $api_access_key = $this->get_setting( 'fixer_api_key' );

        if ( empty( $api_access_key ) ) {
            return $rates;
        }

        $api_url    = 'https://data.fixer.io/api/' . $date . '?access_key=' . $api_access_key;
        $currencies = $this->get_currencies();

        if ( empty( $default_code ) ) {
            if ( ! empty( $currencies[0]['code'] ) ) {
                $default_code = $currencies[0]['code'];
            } else {
                return $rates;
            }
        }

        $default_code = strtoupper( $default_code );

        $api_url .= '&base=' . $default_code;

        if ( empty( $specific_exchange_rates_codes ) ) {
            if ( is_array( $currencies ) && count( $currencies ) > 1 ) {
                foreach ( array_slice( $currencies, 1 ) as $currency ) {
                    if ( ! empty( $currency['code'] ) ) {
                        $specific_exchange_rates_codes[] = strtoupper( $currency['code'] );
                    }
                }
            }
        } else {
            foreach ( $specific_exchange_rates_codes as &$specific_exchange_rates_code ) {
                $specific_exchange_rates_code = strtoupper( $specific_exchange_rates_code );
            }
        }

        if ( ! empty( $specific_exchange_rates_codes ) ) {
            $api_url .= '&symbols=' . implode( ',', $specific_exchange_rates_codes );
        }

        $api_response = wp_remote_get( $api_url );

        if ( is_array( $api_response ) && ! empty( $api_response['body'] ) ) {
            $api_results_json = $api_response['body'];
            $api_results_arr  = json_decode( $api_results_json, true );

            if ( is_array( $api_results_arr ) && ! empty( $api_results_arr['rates'] ) ) {
                $rates = $api_results_arr['rates'];
            }
        }

        return $rates;
    }

    /*
    * Update exchange rates
    */

    function update_exchange_rates() {

        if ( $this->has_auto_conversion() ) {
            $settings = get_option( $this->settings_title );

            if ( isset( $settings['rates_last_update'] ) && ( strtotime( $settings['rates_last_update'] ) <= strtotime( "-1 day" ) ) ) {
                wp_register_script( $this->settings_title . 'update_rates', $this->settings['url'] . 'assets/js/update-rates.js', [ 'jquery' ], $this->settings['version'], false );
                wp_localize_script( $this->settings_title . 'update_rates', 'acfCurrencyUpdateRatesGlobals', array(
                    'ajax_url'       => admin_url( 'admin-ajax.php' ),
                    'nonce'          => wp_create_nonce( 'acf-currency-update-rates-kZV4r2' ),
                    'settings_title' => $this->settings_title,
                ) );
                wp_enqueue_script( $this->settings_title . 'update_rates' );
            }
        }
    }

    function update_exchange_rates_ajax() {

        if ( ! wp_verify_nonce( $_REQUEST['nonce'], 'acf-currency-update-rates-kZV4r2' ) || ! isset( $_REQUEST['settings_title'] ) ) {
            wp_send_json_error();
            exit;
        }

        $settings = get_option( $_REQUEST['settings_title'] );
        $rates    = acf_currency_get_api_exchange_rates();

        if ( ! empty( $rates ) ) {
            $settings['rates'] = $rates;
        }

        $settings['rates_last_update'] = date( 'Y-m-d H:i:s' );

        update_option( $this->settings_title, $settings );

        wp_send_json_success( [
            'rates'             => $settings['rates'],
            'rates_last_update' => $settings['rates_last_update']
        ] );
        exit;
    }
}


// initialize
new acf_plugin_currency();

// class_exists check
endif;

?>
