<?php

class WPC_Tools_Public_Module_Emails extends WPC_Tools_Public_Module {

  const MODULE_NAME = WPC_Tools_Admin_Module_Emails::MODULE_NAME;

  protected static $_instance = null;

  /**
   * WPC Tools Admin Instance.
   *
   * Ensures only one instance of WPC_Tools_Admin is loaded or can be loaded.
   */
  public static function instance() {
    if ( is_null( self::$_instance ) ) {
      self::$_instance = new self();
    }

    return self::$_instance;
  }

  /**
   * WPC_Tools_Module_Emails constructor.
   */
  public function __construct() {
    $this->settings_title = str_replace( '-', '_', self::MODULE_NAME );
    $this->module_hooks();
  }

  /**
   * Load General Module hooks
   */
  private function module_hooks() {

    if ( boolval( wpc_get_setting( 'enable_wp_mail_override', $this->settings_title ) ) ) {
      WPC_Tools::loader()->add_filter( 'wp_mail_content_type', $this, 'get_wp_mail_content_type' );
      WPC_Tools::loader()->add_filter( 'password_change_email', $this, 'get_password_change_email', 10, 3 );
      WPC_Tools::loader()->add_filter( 'email_change_email', $this, 'get_email_change_email', 10, 3 );
      WPC_Tools::loader()->add_filter( 'retrieve_password_message', $this, 'get_retrieve_password_message', 10, 4 );

      WPC_Tools::loader()->add_action( 'edit_user_profile_update', $this, 'before_send_wp_mail' );
      WPC_Tools::loader()->add_action( 'login_form_retrievepassword', $this, 'before_send_wp_mail' );
      WPC_Tools::loader()->add_action( 'login_form_lostpassword', $this, 'before_send_wp_mail' );
      WPC_Tools::loader()->add_action( 'lost_password', $this, 'after_send_wp_mail' );
    }

    if ( boolval( wpc_get_setting( 'enable_gravity_override', $this->settings_title ) ) ) {
      WPC_Tools::loader()->add_filter( 'gform_pre_send_email', $this, 'get_gravity_email_template' );
    }
  }

  /**
   * Get WP Mail Content Type
   *
   * @param $content_type
   *
   * @return string
   */
  public function get_wp_mail_content_type( $content_type ) {

    return apply_filters( 'wpc_mail_content_type', 'text/html' );
  }

  /**
   * Get WP Mail From Email Address
   *
   * @param $original_email_address
   *
   * @return string
   */
  public function get_wp_mail_from( $original_email_address ) {

    $wp_from_address = get_option( 'admin_email' );

    if ( '' === $wp_from_address ) {
      return $original_email_address;
    }

    return $wp_from_address;
  }

  /**
   * Get WP Mail From Name
   *
   * @param $original_email_from
   *
   * @return string
   */
  public function get_wp_mail_from_name( $original_email_from ) {

    $wp_from_name = get_option( 'blogname' );

    if ( '' === $wp_from_name ) {
      return $original_email_from;
    }

    return $wp_from_name;
  }

  /**
   * Get password change notification
   *
   * @param $pass_change_mail
   *
   * @return array
   */
  public function get_password_change_email( $pass_change_email, $user, $userdata ) {

    $wpc_email_arr = apply_filters( 'wpc_email_password_change_email', [
      'subject' => $pass_change_email['subject'],
      'heading' => str_replace( '[%s] ', '', $pass_change_email['subject'] ),
      'message' => nl2br( $pass_change_email['message'] )
    ], $user, $userdata );

    $pass_change_email['subject'] = $wpc_email_arr['subject'];
    $pass_change_email['message'] = wpc_get_the_email_frame( $wpc_email_arr['message'], [ 'email_heading' => $wpc_email_arr['heading'] ] );

    return $pass_change_email;
  }

  /**
   * Get email change notification
   *
   * @param $email_change_mail
   *
   * @return array
   */
  public function get_email_change_email( $email_change_email, $user, $userdata ) {

    $wpc_email_arr = apply_filters( 'wpc_email_email_change_email', [
      'subject' => $email_change_email['subject'],
      'heading' => str_replace( '[%s] ', '', $email_change_email['subject'] ),
      'message' => nl2br( $email_change_email['message'] )
    ], $user, $userdata );

    $email_change_email['message'] = $wpc_email_arr['subject'];
    $email_change_email['message'] = wpc_get_the_email_frame( $wpc_email_arr['message'], [ 'email_heading' => $wpc_email_arr['heading'] ] );

    return $email_change_email;
  }

  /**
   * Get retrieve password message
   *
   * @param $message
   *
   * @return string
   */
  public function get_retrieve_password_message( $message, $key, $user_login, $user_data ) {

    $wpc_email_arr = apply_filters( 'wpc_email_retrieve_password_message', [
      'heading' => str_replace( '[%s] ', '', __( '[%s] Password Reset' ) ),
      'message' => nl2br( preg_replace( '/(\s\s<?((http)+(s)?:\/\/[^<>\s]+)>?)/i', '<a href="$2" target="_blank">$2</a>', $message ) )
    ], $key, $user_login, $user_data );

    return wpc_get_the_email_frame( $wpc_email_arr['message'], [ 'email_heading' => $wpc_email_arr['heading'] ] );
  }

  /**
   * Get Gravity email template
   *
   * @param $email
   *
   * @return array
   */
  public function get_gravity_email_template( $email ) {

    $email['message'] = wpc_get_the_email_frame( $email['message'], [ 'email_heading' => $email['subject'] ] );

    return $email;
  }

  /**
   * Before send wp mail
   */
  public function before_send_wp_mail() {

    add_filter( 'wp_mail_from', array( WPC_Tools_Public_Module_Emails::instance(), 'get_wp_mail_from' ) );
    add_filter( 'wp_mail_from_name', array( WPC_Tools_Public_Module_Emails::instance(), 'get_wp_mail_from_name' ) );
  }

  /**
   * After send wp mail
   */
  public function after_send_wp_mail() {

    remove_filter( 'wp_mail_from', array( WPC_Tools_Public_Module_Emails::instance(), 'get_wp_mail_from' ) );
    remove_filter( 'wp_mail_from_name', array(
      WPC_Tools_Public_Module_Emails::instance(),
      'get_wp_mail_from_name'
    ) );
  }
}
